"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extract = void 0;
var tslib_1 = require("tslib");
var console_utils_1 = require("./console_utils");
var fs_extra_1 = require("fs-extra");
var ts_transformer_1 = require("@formatjs/ts-transformer");
var formatters_1 = require("./formatters");
var json_stable_stringify_1 = tslib_1.__importDefault(require("json-stable-stringify"));
var vue_extractor_1 = require("./vue_extractor");
var parse_script_1 = require("./parse_script");
var printer_1 = require("@formatjs/icu-messageformat-parser/printer");
var manipulator_1 = require("@formatjs/icu-messageformat-parser/manipulator");
var icu_messageformat_parser_1 = require("@formatjs/icu-messageformat-parser");
function calculateLineColFromOffset(text, start) {
    if (!start) {
        return { line: 1, col: 1 };
    }
    var chunk = text.slice(0, start);
    var lines = chunk.split('\n');
    var lastLine = lines[lines.length - 1];
    return { line: lines.length, col: lastLine.length };
}
function processFile(source, fn, _a) {
    var idInterpolationPattern = _a.idInterpolationPattern, opts = tslib_1.__rest(_a, ["idInterpolationPattern"]);
    var messages = [];
    var meta;
    opts = tslib_1.__assign(tslib_1.__assign({}, opts), { additionalComponentNames: tslib_1.__spreadArray([
            '$formatMessage'
        ], (opts.additionalComponentNames || [])), onMsgExtracted: function (_, msgs) {
            if (opts.extractSourceLocation) {
                msgs = msgs.map(function (msg) { return (tslib_1.__assign(tslib_1.__assign({}, msg), calculateLineColFromOffset(source, msg.start))); });
            }
            messages = messages.concat(msgs);
        }, onMetaExtracted: function (_, m) {
            meta = m;
        } });
    if (!opts.overrideIdFn && idInterpolationPattern) {
        opts = tslib_1.__assign(tslib_1.__assign({}, opts), { overrideIdFn: function (id, defaultMessage, description, fileName) {
                return id ||
                    ts_transformer_1.interpolateName({
                        resourcePath: fileName,
                    }, idInterpolationPattern, {
                        content: description
                            ? defaultMessage + "#" + description
                            : defaultMessage,
                    });
            } });
    }
    console_utils_1.debug('Processing opts for %s: %s', fn, opts);
    var scriptParseFn = parse_script_1.parseScript(opts, fn);
    if (fn.endsWith('.vue')) {
        console_utils_1.debug('Processing %s using vue extractor', fn);
        vue_extractor_1.parseFile(source, fn, scriptParseFn);
    }
    else {
        console_utils_1.debug('Processing %s using typescript extractor', fn);
        scriptParseFn(source);
    }
    console_utils_1.debug('Done extracting %s messages: %s', fn, messages);
    if (meta) {
        console_utils_1.debug('Extracted meta:', meta);
        messages.forEach(function (m) { return (m.meta = meta); });
    }
    return { messages: messages, meta: meta };
}
/**
 * Extract strings from source files
 * @param files list of files
 * @param extractOpts extract options
 * @returns messages serialized as JSON string since key order
 * matters for some `format`
 */
function extract(files, extractOpts) {
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var throws, readFromStdin, flatten, opts, rawResults, stdinSource, formatter, extractionResults, extractedMessages, _i, extractionResults_1, messages_3, _a, messages_1, message, id, description, defaultMessage, error, existing, error, results, messages, _b, messages_2, _c, id, msg;
        var _this = this;
        return tslib_1.__generator(this, function (_d) {
            switch (_d.label) {
                case 0:
                    throws = extractOpts.throws, readFromStdin = extractOpts.readFromStdin, flatten = extractOpts.flatten, opts = tslib_1.__rest(extractOpts, ["throws", "readFromStdin", "flatten"]);
                    if (!readFromStdin) return [3 /*break*/, 2];
                    console_utils_1.debug("Reading input from stdin");
                    // Read from stdin
                    if (process.stdin.isTTY) {
                        console_utils_1.warn('Reading source file from TTY.');
                    }
                    return [4 /*yield*/, console_utils_1.getStdinAsString()];
                case 1:
                    stdinSource = _d.sent();
                    rawResults = [processFile(stdinSource, 'dummy', opts)];
                    return [3 /*break*/, 4];
                case 2: return [4 /*yield*/, Promise.all(files.map(function (fn) { return tslib_1.__awaiter(_this, void 0, void 0, function () {
                        var source, e_1;
                        return tslib_1.__generator(this, function (_a) {
                            switch (_a.label) {
                                case 0:
                                    console_utils_1.debug('Extracting file:', fn);
                                    _a.label = 1;
                                case 1:
                                    _a.trys.push([1, 3, , 4]);
                                    return [4 /*yield*/, fs_extra_1.readFile(fn, 'utf8')];
                                case 2:
                                    source = _a.sent();
                                    return [2 /*return*/, processFile(source, fn, opts)];
                                case 3:
                                    e_1 = _a.sent();
                                    if (throws) {
                                        throw e_1;
                                    }
                                    else {
                                        console_utils_1.warn(e_1);
                                    }
                                    return [3 /*break*/, 4];
                                case 4: return [2 /*return*/];
                            }
                        });
                    }); }))];
                case 3:
                    rawResults = _d.sent();
                    _d.label = 4;
                case 4: return [4 /*yield*/, formatters_1.resolveBuiltinFormatter(opts.format)];
                case 5:
                    formatter = _d.sent();
                    extractionResults = rawResults.filter(function (r) { return !!r; });
                    extractedMessages = new Map();
                    for (_i = 0, extractionResults_1 = extractionResults; _i < extractionResults_1.length; _i++) {
                        messages_3 = extractionResults_1[_i].messages;
                        for (_a = 0, messages_1 = messages_3; _a < messages_1.length; _a++) {
                            message = messages_1[_a];
                            id = message.id, description = message.description, defaultMessage = message.defaultMessage;
                            if (!id) {
                                error = new Error("[FormatJS CLI] Missing message id for message: \n" + JSON.stringify(message, undefined, 2));
                                if (throws) {
                                    throw error;
                                }
                                else {
                                    console_utils_1.warn(error.message);
                                }
                                continue;
                            }
                            if (extractedMessages.has(id)) {
                                existing = extractedMessages.get(id);
                                if (description !== existing.description ||
                                    defaultMessage !== existing.defaultMessage) {
                                    error = new Error("[FormatJS CLI] Duplicate message id: \"" + id + "\", " +
                                        'but the `description` and/or `defaultMessage` are different.');
                                    if (throws) {
                                        throw error;
                                    }
                                    else {
                                        console_utils_1.warn(error.message);
                                    }
                                }
                            }
                            extractedMessages.set(id, message);
                        }
                    }
                    results = {};
                    messages = Array.from(extractedMessages.values());
                    for (_b = 0, messages_2 = messages; _b < messages_2.length; _b++) {
                        _c = messages_2[_b];
                        id = _c.id, msg = tslib_1.__rest(_c, ["id"]);
                        if (flatten && msg.defaultMessage) {
                            msg.defaultMessage = printer_1.printAST(manipulator_1.hoistSelectors(icu_messageformat_parser_1.parse(msg.defaultMessage)));
                        }
                        results[id] = msg;
                    }
                    return [2 /*return*/, json_stable_stringify_1.default(formatter.format(results), {
                            space: 2,
                            cmp: formatter.compareMessages || undefined,
                        })];
            }
        });
    });
}
exports.extract = extract;
/**
 * Extract strings from source files, also writes to a file.
 * @param files list of files
 * @param extractOpts extract options
 * @returns A Promise that resolves if output file was written successfully
 */
function extractAndWrite(files, extractOpts) {
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var outFile, opts, serializedResult;
        return tslib_1.__generator(this, function (_a) {
            switch (_a.label) {
                case 0:
                    outFile = extractOpts.outFile, opts = tslib_1.__rest(extractOpts, ["outFile"]);
                    return [4 /*yield*/, extract(files, opts)];
                case 1:
                    serializedResult = (_a.sent()) + '\n';
                    if (outFile) {
                        console_utils_1.debug('Writing output file:', outFile);
                        return [2 /*return*/, fs_extra_1.outputFile(outFile, serializedResult)];
                    }
                    process.stdout.write(serializedResult);
                    return [2 /*return*/];
            }
        });
    });
}
exports.default = extractAndWrite;
