"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.compile = void 0;
var tslib_1 = require("tslib");
var icu_messageformat_parser_1 = require("@formatjs/icu-messageformat-parser");
var fs_extra_1 = require("fs-extra");
var json_stable_stringify_1 = tslib_1.__importDefault(require("json-stable-stringify"));
var console_utils_1 = require("./console_utils");
var formatters_1 = require("./formatters");
var pseudo_locale_1 = require("./pseudo_locale");
/**
 * Aggregate `inputFiles` into a single JSON blob and compile.
 * Also checks for conflicting IDs.
 * Then returns the serialized result as a `string` since key order
 * makes a difference in some vendor.
 * @param inputFiles Input files
 * @param opts Options
 * @returns serialized result in string format
 */
function compile(inputFiles, opts) {
    if (opts === void 0) { opts = {}; }
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var ast, format, pseudoLocale, skipErrors, formatter, messages, messageAsts, idsWithFileName, compiledFiles, i, inputFile, compiled, id, msgAst;
        return tslib_1.__generator(this, function (_a) {
            switch (_a.label) {
                case 0:
                    console_utils_1.debug('Compiling files:', inputFiles);
                    ast = opts.ast, format = opts.format, pseudoLocale = opts.pseudoLocale, skipErrors = opts.skipErrors;
                    return [4 /*yield*/, formatters_1.resolveBuiltinFormatter(format)];
                case 1:
                    formatter = _a.sent();
                    messages = {};
                    messageAsts = {};
                    idsWithFileName = {};
                    return [4 /*yield*/, Promise.all(inputFiles.map(function (f) { return fs_extra_1.readJSON(f).then(formatter.compile); }))];
                case 2:
                    compiledFiles = _a.sent();
                    console_utils_1.debug('Compiled files:', compiledFiles);
                    for (i = 0; i < inputFiles.length; i++) {
                        inputFile = inputFiles[i];
                        console_utils_1.debug('Processing file:', inputFile);
                        compiled = compiledFiles[i];
                        for (id in compiled) {
                            if (messages[id] && messages[id] !== compiled[id]) {
                                throw new Error("Conflicting ID \"" + id + "\" with different translation found in these 2 files:\nID: " + id + "\nMessage from " + idsWithFileName[id] + ": " + messages[id] + "\nMessage from " + compiled[id] + ": " + inputFile + "\n");
                            }
                            try {
                                msgAst = icu_messageformat_parser_1.parse(compiled[id]);
                                messages[id] = compiled[id];
                                switch (pseudoLocale) {
                                    case 'xx-LS':
                                        messageAsts[id] = pseudo_locale_1.generateXXLS(msgAst);
                                        break;
                                    case 'xx-AC':
                                        messageAsts[id] = pseudo_locale_1.generateXXAC(msgAst);
                                        break;
                                    case 'xx-HA':
                                        messageAsts[id] = pseudo_locale_1.generateXXHA(msgAst);
                                        break;
                                    case 'en-XA':
                                        messageAsts[id] = pseudo_locale_1.generateENXA(msgAst);
                                        break;
                                    case 'en-XB':
                                        messageAsts[id] = pseudo_locale_1.generateENXB(msgAst);
                                        break;
                                    default:
                                        messageAsts[id] = msgAst;
                                        break;
                                }
                                idsWithFileName[id] = inputFile;
                            }
                            catch (e) {
                                console_utils_1.warn('Error validating message "%s" with ID "%s" in file "%s"', compiled[id], id, inputFile);
                                if (!skipErrors) {
                                    throw e;
                                }
                            }
                        }
                    }
                    return [2 /*return*/, json_stable_stringify_1.default(ast ? messageAsts : messages, {
                            space: 2,
                            cmp: formatter.compareMessages || undefined,
                        })];
            }
        });
    });
}
exports.compile = compile;
/**
 * Aggregate `inputFiles` into a single JSON blob and compile.
 * Also checks for conflicting IDs and write output to `outFile`.
 * @param inputFiles Input files
 * @param compileOpts options
 * @returns A `Promise` that resolves if file was written successfully
 */
function compileAndWrite(inputFiles, compileOpts) {
    if (compileOpts === void 0) { compileOpts = {}; }
    return tslib_1.__awaiter(this, void 0, void 0, function () {
        var outFile, opts, serializedResult;
        return tslib_1.__generator(this, function (_a) {
            switch (_a.label) {
                case 0:
                    outFile = compileOpts.outFile, opts = tslib_1.__rest(compileOpts, ["outFile"]);
                    return [4 /*yield*/, compile(inputFiles, opts)];
                case 1:
                    serializedResult = _a.sent();
                    if (outFile) {
                        console_utils_1.debug('Writing output file:', outFile);
                        return [2 /*return*/, fs_extra_1.outputFile(outFile, serializedResult)];
                    }
                    process.stdout.write(serializedResult);
                    process.stdout.write('\n');
                    return [2 /*return*/];
            }
        });
    });
}
exports.default = compileAndWrite;
