// React & Next.js
import React, { FormEvent, useRef, useState } from 'react'
import axios, { AxiosRequestConfig } from 'axios'
import { useRouter } from 'next/router'
import ErrorComponent from 'components/shared/Error'
import { useProductContext } from 'components/Products'
import { token } from 'morgan'
import { login } from 'helpers/auth'
// Application

interface IToken {
  name: string
  token: string
}

const LoginPage = () => {
  const [username, setUsername] = useState('')
  const [password, setPassword] = useState('')
  const [tokens, setTokens] = useState([])
  const [isError, setIsError] = useState(false)
  const { setOffline, setToken } = useProductContext()
  const router = useRouter()
  const tokenSelector = useRef(null)

  const handleSubmit = async (e: FormEvent) => {
    e.preventDefault()
    login(username, password)
      .then((receivedTokens: IToken[]) => {

        // cleanup
        setUsername('')
        setPassword('')
        setIsError(false)

        if (receivedTokens.length === 1) {
          setToken(receivedTokens[0].token)
          localStorage.setItem('token', receivedTokens[0].token)
          router.push('/').then(() => {
            router.reload()
          })
        } else {
          // multiple token
          setTokens(receivedTokens)
          tokenSelector.current.classList.remove('hidden')
        }
      })
      .catch((err) => {
        // console.log(err)
        setIsError(true)
      })
  }

  return (
    <>
      <section className="flex flex-col items-center h-screen md:flex-row justify-center">
        <div className="hidden w-full h-screen bg-no-repeat bg-cover bg-login-texture lg:block md:w-1/2 xl:w-2/3"></div>
        <div className="flex items-center justify-center w-full h-screen px-6 bg-white md:max-w-md lg:max-w-full md:mx-0 md:w-1/2 xl:w-1/3 lg:px-16 xl:px-12z">
          <div className="w-full h-100">
            <ErrorComponent isError={isError} message="نام کاربری یا کلمه عبور اشتباه است" />
            <h1 className="mt-12 text-xl font-bold leading-tight md:text-2xl">ورود به حساب کاربری</h1>

            <form className="mt-6" onSubmit={(e) => handleSubmit(e)}>
              <div>
                <label className="block text-gray-700">نام کاربری</label>
                <input
                  type="text"
                  className="w-full px-4 py-3 mt-2 bg-gray-200 border rounded-lg focus:border-blue-500 focus:bg-white focus:outline-none"
                  autoFocus
                  value={username}
                  onChange={(e) => {
                    setUsername(e.target.value)
                  }}
                  autoComplete=""
                  required
                />
              </div>

              <div className="mt-4">
                <label className="block text-gray-700">کلمه عبور</label>
                <input
                  type="password"
                  value={password}
                  onChange={(e) => setPassword(e.target.value)}
                  name=""
                  id=""
                  className="w-full px-4 py-3 mt-2 bg-gray-200 border rounded-lg focus:border-blue-500 focus:bg-white focus:outline-none"
                  required
                />
              </div>

              <div className="mt-2 text-right">
                <a href="#" className="text-sm font-semibold text-gray-700 hover:text-blue-700 focus:text-blue-700">
                  کلمه عبور را فراموش کرده اید؟
                </a>
              </div>

              <button type="submit" className="mt-8 btn btn-secondary btn-block">
                ورود
              </button>
              <div ref={tokenSelector} className="hidden w-full mt-2 dropdown dropdown-top dropdown-end">
                <div tabIndex={0} className="w-full m-1 btn btn-accent">
                  انتخاب توکن
                </div>
                <ul tabIndex={0} className="w-full p-2 shadow menu dropdown-content bg-base-100 rounded-box">
                  {tokens &&
                    tokens.map((token) => (
                      <li key={token.token}>
                        <a
                          onClick={() => {
                            setToken(token.token)
                            localStorage.setItem('token', token.token)
                            router.push('/').then(() => {
                              router.reload()
                            })
                          }}>
                          {token.name}
                        </a>
                      </li>
                    ))}
                </ul>
              </div>
              <button
                className="w-full mt-8 btn btn-primary"
                onClick={() => {
                  setOffline(true)
                  router.push('/')
                }}>
                آفلاین
              </button>
            </form>
          </div>
        </div>
      </section>
    </>
  )
}

export default LoginPage
