import CustomerSuggest from 'components/CustomerSuggest'
import { useProductContext } from 'components/Products'
import ErrorComponent from 'components/shared/Error'
import ImageComponent from 'components/shared/image-component'
import { cartTotalPrice, formatPrice } from 'helpers/format'
import { addItemToCart, removeItemFromCart, removeItemsFromCart, updateOrderNote } from 'helpers/check-items'
import ICustomer from 'interfaces/customer'
import Link from 'next/link'
import { useEffect, useState } from 'react'
import { submitOrder } from 'services/order'

const CartPage = () => {
    const { order, setOrder, token } = useProductContext()
    const [isError, setIsError] = useState(false)
    const [note, setNote] = useState('')
    const [tableId, setTableId] = useState('')
    const [orderTypeDefault, setOrderTypeDefault] = useState(null)

    useEffect(() => {
        if (orderTypeDefault === null) {
            setOrderTypeDefault(localStorage.getItem('order_type_default') ? localStorage.getItem('order_type_default') : 'EXP')
        }
    })

    enum errorMessages {
        Customer = 'لطفا ابتدا مشتری را ثبت نمایید',
        NoServer = 'ارتباط با سرور داخلی برقرار نیست',
        // TableId = 'لطفا ابتدا شماره میز را ثبت نمایید',
    }

    const [errorMessage, setErrorMessage] = useState(errorMessages.Customer)
    const [customer, setCustomer] = useState<ICustomer>({ name: '', id: -1 })
    const [userPoint, setUserPoint] = useState(0);

    useEffect(() => {
        // console.log(order)
    }, [order])

    return (
        <>
            <section className="flex flex-col h-screen py-28">
                <main className="w-full my-8">
                    <div className="w-full pr-8 mb-4 text-yellow-700">
                        <Link href="/">
                            <a className="cursor-pointer">بازگشت</a>
                        </Link>
                    </div>
                    <ErrorComponent isError={isError} message={errorMessage} />
                    <div className="flex flex-col items-center justify-center w-full h-full">
                        {order.length === 0 && (
                            <div
                                className="flex flex-col items-center justify-center h-full p-24 text-2xl text-red-400 bg-yellow-50 rounded-3xl">
                                <ImageComponent src="/images/empty.png" alt="empty basket" layout="fixed" width={100}
                                    height={100} />
                                <p>سبد خرید شما خالی است</p>
                            </div>
                        )}
                        {order.length > 0 && (
                            <>
                                <CustomerSuggest customer={customer} setCustomer={setCustomer} setIsError={setIsError} setUserPoint={setUserPoint} setUserDiscount={''} />

                                <div className="w-full px-16 mt-4 overflow-x-auto ">
                                    <table className="table w-full text-lg table-zebra">
                                        <thead>
                                            <tr>
                                                <th></th>
                                                <th>نام</th>
                                                <th>قیمت</th>
                                                <th>تعداد</th>
                                                <th>قیمت کل</th>
                                                <th>یادداشت</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            {order.map((item) => (
                                                <tr key={item.product.id}>
                                                    <th>
                                                        <button
                                                            className="btn btn-circle btn-secondary btn-xs"
                                                            onClick={() => {
                                                                setOrder([...removeItemsFromCart(item.product.unique, order)])
                                                            }}>
                                                            <svg
                                                                xmlns="http://www.w3.org/2000/svg"
                                                                fill="none"
                                                                viewBox="0 0 24 24"
                                                                className="inline-block w-4 h-4 stroke-current">
                                                                <path strokeLinecap="round" strokeLinejoin="round"
                                                                    strokeWidth="2" d="M6 18L18 6M6 6l12 12"></path>
                                                            </svg>
                                                        </button>
                                                    </th>
                                                    <td>{item.product.name}</td>
                                                    <td>{formatPrice(item.product.price)}</td>
                                                    <td>
                                                        <div className="flex">
                                                            <span
                                                                className="cursor-pointer"
                                                                onClick={() => {
                                                                    setOrder([...removeItemFromCart(item.product.unique, order)])
                                                                }}>
                                                                <ImageComponent src="/images/minus.png" width={24} height={24} alt="minus" />
                                                            </span>
                                                            <span className="px-4">{item.count}</span>
                                                            <span
                                                                className="cursor-pointer"
                                                                onClick={() => {
                                                                    setOrder([...addItemToCart(item.product.unique, order)])
                                                                }}>
                                                                <ImageComponent src="/images/plus.png" width={24} height={24} alt="plus" />
                                                            </span>
                                                        </div>
                                                    </td>
                                                    <td>{formatPrice(item.count * item.product.price)}</td>
                                                    <td>
                                                        <input
                                                            type="text"
                                                            placeholder="توضیحات"
                                                            className="w-full border-2 border-gray-300 input"
                                                            onChange={(e) => {
                                                                setOrder([...updateOrderNote(item.product.unique, order, e.currentTarget.value)])
                                                            }}
                                                        />
                                                    </td>
                                                </tr>
                                            ))}
                                        </tbody>
                                    </table>

                                    <div className="flex flex-col p-4 my-8 bg-yellow-100 rounded-2xl">
                                        <div className="flex items-center mb-3">
                                            <div className="flex flex-col">
                                                <p>{cartTotalPrice(order)}</p>
                                                <p>{customer.name}</p>
                                            </div>
                                            <div className={'inline ml-4 ring-2'}>
                                                <select defaultValue={orderTypeDefault} onChange={(e) => {
                                                    setOrderTypeDefault(e.target.value)
                                                }}>
                                                    <option value={1}>EXP</option>
                                                    <option value={2}>Delivery</option>
                                                    <option value={3}>Salon</option>
                                                </select>
                                            </div>
                                            <button
                                                onClick={() => {
                                                    if (!customer || customer.id < 0) {
                                                        setErrorMessage(errorMessages.Customer)
                                                        setIsError(true)
                                                        return
                                                    }
                                                    // if (!tableId) {
                                                    //   setErrorMessage(errorMessages.TableId)
                                                    //   setIsError(true)
                                                    //   return
                                                    // }
                                                    setIsError(false)
                                                    setErrorMessage(errorMessages.NoServer)
                                                    // submitOrder(order, token, customer, note, tableId, orderTypeDefault)
                                                    //     .then(() => {
                                                    //         setCustomer({name: '', id: -1})
                                                    //         setOrder([])
                                                    //         setTableId('')
                                                    //         setNote('')
                                                    //         setOrderTypeDefault(localStorage.getItem('order_type_default') ? localStorage.getItem('order_type_default') : '3')
                                                    //     })
                                                    //     .catch(() => {
                                                    //         setIsError(true)
                                                    //     })
                                                }}
                                                className="w-24 mr-8 btn-accent btn btn-xs md:btn-sm lg:btn-md">
                                                ثبت
                                            </button>
                                        </div>
                                        <input
                                            type={'number'}
                                            className="h-4 textarea textarea-bordered mb-2"
                                            value={tableId}
                                            onChange={(e) => {
                                                setTableId(e.target.value)
                                            }}
                                            placeholder="شماره میز" />
                                        <textarea
                                            className="h-4 textarea textarea-bordered"
                                            value={note}
                                            onChange={(e) => {
                                                setNote(e.target.value)
                                            }}
                                            placeholder="توضیحات" />
                                    </div>
                                </div>
                            </>
                        )}
                    </div>
                </main>
            </section>
        </>
    )
}

export default CartPage
