"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.storeMessage = exports.wasExtracted = exports.tagAsExtracted = exports.evaluateMessageDescriptor = exports.createMessageDescriptor = exports.getMessageDescriptorKey = void 0;
var tslib_1 = require("tslib");
var icu_messageformat_parser_1 = require("@formatjs/icu-messageformat-parser");
var ts_transformer_1 = require("@formatjs/ts-transformer");
var DESCRIPTOR_PROPS = new Set(['id', 'description', 'defaultMessage']);
function evaluatePath(path) {
    var evaluated = path.evaluate();
    if (evaluated.confident) {
        return evaluated.value;
    }
    throw path.buildCodeFrameError('[React Intl] Messages must be statically evaluate-able for extraction.');
}
function getMessageDescriptorKey(path) {
    if (path.isIdentifier() || path.isJSXIdentifier()) {
        return path.node.name;
    }
    return evaluatePath(path);
}
exports.getMessageDescriptorKey = getMessageDescriptorKey;
function getMessageDescriptorValue(path) {
    if (!path) {
        return '';
    }
    if (path.isJSXExpressionContainer()) {
        path = path.get('expression');
    }
    // Always trim the Message Descriptor values.
    var descriptorValue = evaluatePath(path);
    return descriptorValue;
}
function createMessageDescriptor(propPaths) {
    return propPaths.reduce(function (hash, _a) {
        var keyPath = _a[0], valuePath = _a[1];
        var key = getMessageDescriptorKey(keyPath);
        if (DESCRIPTOR_PROPS.has(key)) {
            hash[key] = valuePath;
        }
        return hash;
    }, {
        id: undefined,
        defaultMessage: undefined,
        description: undefined,
    });
}
exports.createMessageDescriptor = createMessageDescriptor;
function evaluateMessageDescriptor(descriptorPath, isJSXSource, filename, idInterpolationPattern, overrideIdFn, preserveWhitespace) {
    if (isJSXSource === void 0) { isJSXSource = false; }
    var id = getMessageDescriptorValue(descriptorPath.id);
    var defaultMessage = getICUMessageValue(descriptorPath.defaultMessage, {
        isJSXSource: isJSXSource,
    }, preserveWhitespace);
    var description = getMessageDescriptorValue(descriptorPath.description);
    if (overrideIdFn) {
        id = overrideIdFn(id, defaultMessage, description, filename);
    }
    else if (!id && idInterpolationPattern && defaultMessage) {
        id = ts_transformer_1.interpolateName({ resourcePath: filename }, idInterpolationPattern, {
            content: description
                ? defaultMessage + "#" + description
                : defaultMessage,
        });
    }
    var descriptor = {
        id: id,
    };
    if (description) {
        descriptor.description = description;
    }
    if (defaultMessage) {
        descriptor.defaultMessage = defaultMessage;
    }
    return descriptor;
}
exports.evaluateMessageDescriptor = evaluateMessageDescriptor;
function getICUMessageValue(messagePath, _a, preserveWhitespace) {
    var _b = _a === void 0 ? {} : _a, _c = _b.isJSXSource, isJSXSource = _c === void 0 ? false : _c;
    if (!messagePath) {
        return '';
    }
    var message = getMessageDescriptorValue(messagePath);
    if (!preserveWhitespace) {
        message = message.trim().replace(/\s+/gm, ' ');
    }
    try {
        icu_messageformat_parser_1.parse(message);
    }
    catch (parseError) {
        if (isJSXSource &&
            messagePath.isLiteral() &&
            message.indexOf('\\\\') >= 0) {
            throw messagePath.buildCodeFrameError('[React Intl] Message failed to parse. ' +
                'It looks like `\\`s were used for escaping, ' +
                "this won't work with JSX string literals. " +
                'Wrap with `{}`. ' +
                'See: http://facebook.github.io/react/docs/jsx-gotchas.html');
        }
        throw messagePath.buildCodeFrameError('[React Intl] Message failed to parse. ' +
            'See: https://formatjs.io/docs/core-concepts/icu-syntax' +
            ("\n" + parseError));
    }
    return message;
}
var EXTRACTED = Symbol('FormatJSExtracted');
/**
 * Tag a node as extracted
 * Store this in the node itself so that multiple passes work. Specifically
 * if we remove `description` in the 1st pass, 2nd pass will fail since
 * it expect `description` to be there.
 * HACK: We store this in the node instance since this persists across
 * multiple plugin runs
 * @param path
 */
function tagAsExtracted(path) {
    path.node[EXTRACTED] = true;
}
exports.tagAsExtracted = tagAsExtracted;
/**
 * Check if a node was extracted
 * @param path
 */
function wasExtracted(path) {
    return !!path.node[EXTRACTED];
}
exports.wasExtracted = wasExtracted;
/**
 * Store a message in our global messages
 * @param messageDescriptor
 * @param path
 * @param opts
 * @param filename
 * @param messages
 */
function storeMessage(_a, path, _b, filename, messages) {
    var id = _a.id, description = _a.description, defaultMessage = _a.defaultMessage;
    var extractSourceLocation = _b.extractSourceLocation;
    if (!id && !defaultMessage) {
        throw path.buildCodeFrameError('[React Intl] Message Descriptors require an `id` or `defaultMessage`.');
    }
    var loc = {};
    if (extractSourceLocation) {
        loc = tslib_1.__assign({ file: filename }, path.node.loc);
    }
    messages.push(tslib_1.__assign({ id: id, description: description, defaultMessage: defaultMessage }, loc));
}
exports.storeMessage = storeMessage;
