"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.transform = exports.transformWithTs = void 0;
var tslib_1 = require("tslib");
var typescript = tslib_1.__importStar(require("typescript"));
var interpolate_name_1 = require("./interpolate-name");
var icu_messageformat_parser_1 = require("@formatjs/icu-messageformat-parser");
var console_utils_1 = require("./console_utils");
var MESSAGE_DESC_KEYS = [
    'id',
    'defaultMessage',
    'description',
];
function primitiveToTSNode(factory, v) {
    return typeof v === 'string'
        ? factory.createStringLiteral(v)
        : typeof v === 'number'
            ? factory.createNumericLiteral(v + '')
            : typeof v === 'boolean'
                ? v
                    ? factory.createTrue()
                    : factory.createFalse()
                : undefined;
}
function isValidIdentifier(k) {
    try {
        new Function("return {" + k + ":1}");
        return true;
    }
    catch (e) {
        return false;
    }
}
function objToTSNode(factory, obj) {
    if (typeof obj === 'object' && !obj) {
        return factory.createNull();
    }
    var props = Object.entries(obj)
        .filter(function (_a) {
        var _ = _a[0], v = _a[1];
        return typeof v !== 'undefined';
    })
        .map(function (_a) {
        var k = _a[0], v = _a[1];
        return factory.createPropertyAssignment(isValidIdentifier(k) ? k : factory.createStringLiteral(k), primitiveToTSNode(factory, v) ||
            (Array.isArray(v)
                ? factory.createArrayLiteralExpression(v
                    .filter(function (n) { return typeof n !== 'undefined'; })
                    .map(function (n) { return objToTSNode(factory, n); }))
                : objToTSNode(factory, v)));
    });
    return factory.createObjectLiteralExpression(props);
}
function messageASTToTSNode(factory, ast) {
    return factory.createArrayLiteralExpression(ast.map(function (el) { return objToTSNode(factory, el); }));
}
var DEFAULT_OPTS = {
    onMsgExtracted: function () { return undefined; },
    onMetaExtracted: function () { return undefined; },
};
function isMultipleMessageDecl(ts, node) {
    return (ts.isIdentifier(node.expression) &&
        node.expression.text === 'defineMessages');
}
function isSingularMessageDecl(ts, node, additionalComponentNames) {
    var compNames = new Set(tslib_1.__spreadArray([
        'FormattedMessage',
        'defineMessage',
        'formatMessage',
        '$formatMessage'
    ], additionalComponentNames));
    var fnName = '';
    if (ts.isCallExpression(node) && ts.isIdentifier(node.expression)) {
        fnName = node.expression.text;
    }
    else if (ts.isJsxOpeningElement(node) && ts.isIdentifier(node.tagName)) {
        fnName = node.tagName.text;
    }
    else if (ts.isJsxSelfClosingElement(node) &&
        ts.isIdentifier(node.tagName)) {
        fnName = node.tagName.text;
    }
    return compNames.has(fnName);
}
function evaluateStringConcat(ts, node) {
    var right = node.right, left = node.left;
    if (!ts.isStringLiteral(right)) {
        return ['', false];
    }
    if (ts.isStringLiteral(left)) {
        return [left.text + right.text, true];
    }
    if (ts.isBinaryExpression(left)) {
        var _a = evaluateStringConcat(ts, left), result = _a[0], isStatic = _a[1];
        return [result + right.text, isStatic];
    }
    return ['', false];
}
function extractMessageDescriptor(ts, node, _a, sf) {
    var overrideIdFn = _a.overrideIdFn, extractSourceLocation = _a.extractSourceLocation, preserveWhitespace = _a.preserveWhitespace;
    var properties = undefined;
    if (ts.isObjectLiteralExpression(node)) {
        properties = node.properties;
    }
    else if (ts.isJsxOpeningElement(node) || ts.isJsxSelfClosingElement(node)) {
        properties = node.attributes.properties;
    }
    var msg = { id: '' };
    if (!properties) {
        return;
    }
    properties.forEach(function (prop) {
        var name = prop.name;
        var initializer = ts.isPropertyAssignment(prop) || ts.isJsxAttribute(prop)
            ? prop.initializer
            : undefined;
        if (name && ts.isIdentifier(name) && initializer) {
            // <FormattedMessage foo={'barbaz'} />
            if (ts.isJsxExpression(initializer) &&
                initializer.expression &&
                ts.isStringLiteral(initializer.expression)) {
                switch (name.text) {
                    case 'id':
                        msg.id = initializer.expression.text;
                        break;
                    case 'defaultMessage':
                        msg.defaultMessage = initializer.expression.text;
                        break;
                    case 'description':
                        msg.description = initializer.expression.text;
                        break;
                }
            }
            // {id: 'id'}
            else if (ts.isStringLiteral(initializer)) {
                switch (name.text) {
                    case 'id':
                        msg.id = initializer.text;
                        break;
                    case 'defaultMessage':
                        msg.defaultMessage = initializer.text;
                        break;
                    case 'description':
                        msg.description = initializer.text;
                        break;
                }
            }
            // {id: `id`}
            else if (ts.isNoSubstitutionTemplateLiteral(initializer)) {
                switch (name.text) {
                    case 'id':
                        msg.id = initializer.text;
                        break;
                    case 'defaultMessage':
                        msg.defaultMessage = initializer.text;
                        break;
                    case 'description':
                        msg.description = initializer.text;
                        break;
                }
            }
            else if (ts.isJsxExpression(initializer) && initializer.expression) {
                // <FormattedMessage foo={`bar`} />
                if (ts.isNoSubstitutionTemplateLiteral(initializer.expression)) {
                    var expression = initializer.expression;
                    switch (name.text) {
                        case 'id':
                            msg.id = expression.text;
                            break;
                        case 'defaultMessage':
                            msg.defaultMessage = expression.text;
                            break;
                        case 'description':
                            msg.description = expression.text;
                            break;
                    }
                }
                // <FormattedMessage foo={'bar' + 'baz'} />
                else if (ts.isBinaryExpression(initializer.expression)) {
                    var expression = initializer.expression;
                    var _a = evaluateStringConcat(ts, expression), result = _a[0], isStatic = _a[1];
                    if (isStatic) {
                        switch (name.text) {
                            case 'id':
                                msg.id = result;
                                break;
                            case 'defaultMessage':
                                msg.defaultMessage = result;
                                break;
                            case 'description':
                                msg.description = result;
                                break;
                        }
                    }
                }
            }
            // {defaultMessage: 'asd' + bar'}
            else if (ts.isBinaryExpression(initializer)) {
                var _b = evaluateStringConcat(ts, initializer), result = _b[0], isStatic = _b[1];
                if (isStatic) {
                    switch (name.text) {
                        case 'id':
                            msg.id = result;
                            break;
                        case 'defaultMessage':
                            msg.defaultMessage = result;
                            break;
                        case 'description':
                            msg.description = result;
                            break;
                    }
                }
            }
        }
    });
    // We extracted nothing
    if (!msg.defaultMessage && !msg.id) {
        return;
    }
    if (msg.defaultMessage && !preserveWhitespace) {
        msg.defaultMessage = msg.defaultMessage.trim().replace(/\s+/gm, ' ');
    }
    if (msg.defaultMessage && overrideIdFn) {
        switch (typeof overrideIdFn) {
            case 'string':
                if (!msg.id) {
                    msg.id = interpolate_name_1.interpolateName({ resourcePath: sf.fileName }, overrideIdFn, {
                        content: msg.description
                            ? msg.defaultMessage + "#" + msg.description
                            : msg.defaultMessage,
                    });
                }
                break;
            case 'function':
                msg.id = overrideIdFn(msg.id, msg.defaultMessage, msg.description, sf.fileName);
                break;
        }
    }
    if (extractSourceLocation) {
        return tslib_1.__assign(tslib_1.__assign({}, msg), { file: sf.fileName, start: node.pos, end: node.end });
    }
    return msg;
}
/**
 * Check if node is `foo.bar.formatMessage` node
 * @param node
 * @param sf
 */
function isMemberMethodFormatMessageCall(ts, node, additionalFunctionNames) {
    var fnNames = new Set(tslib_1.__spreadArray([
        'formatMessage',
        '$formatMessage'
    ], additionalFunctionNames));
    var method = node.expression;
    // Handle foo.formatMessage()
    if (ts.isPropertyAccessExpression(method)) {
        return fnNames.has(method.name.text);
    }
    // Handle formatMessage()
    return ts.isIdentifier(method) && fnNames.has(method.text);
}
function extractMessageFromJsxComponent(ts, factory, node, opts, sf) {
    var onMsgExtracted = opts.onMsgExtracted;
    if (!isSingularMessageDecl(ts, node, opts.additionalComponentNames || [])) {
        return node;
    }
    var msg = extractMessageDescriptor(ts, node, opts, sf);
    if (!msg) {
        return node;
    }
    if (typeof onMsgExtracted === 'function') {
        onMsgExtracted(sf.fileName, [msg]);
    }
    var newProps = generateNewProperties(ts, factory, node.attributes, {
        defaultMessage: opts.removeDefaultMessage
            ? undefined
            : msg.defaultMessage,
        id: msg.id,
    }, opts.ast);
    if (ts.isJsxOpeningElement(node)) {
        return factory.updateJsxOpeningElement(node, node.tagName, node.typeArguments, factory.createJsxAttributes(newProps));
    }
    return factory.updateJsxSelfClosingElement(node, node.tagName, node.typeArguments, factory.createJsxAttributes(newProps));
}
function setAttributesInObject(ts, factory, node, msg, ast) {
    var newProps = tslib_1.__spreadArray([
        factory.createPropertyAssignment('id', factory.createStringLiteral(msg.id))
    ], (msg.defaultMessage
        ? [
            factory.createPropertyAssignment('defaultMessage', ast
                ? messageASTToTSNode(factory, icu_messageformat_parser_1.parse(msg.defaultMessage))
                : factory.createStringLiteral(msg.defaultMessage)),
        ]
        : []));
    for (var _i = 0, _a = node.properties; _i < _a.length; _i++) {
        var prop = _a[_i];
        if (ts.isPropertyAssignment(prop) &&
            ts.isIdentifier(prop.name) &&
            MESSAGE_DESC_KEYS.includes(prop.name.text)) {
            continue;
        }
        if (ts.isPropertyAssignment(prop)) {
            newProps.push(prop);
        }
    }
    return factory.createObjectLiteralExpression(factory.createNodeArray(newProps));
}
function generateNewProperties(ts, factory, node, msg, ast) {
    var newProps = tslib_1.__spreadArray([
        factory.createJsxAttribute(factory.createIdentifier('id'), factory.createStringLiteral(msg.id))
    ], (msg.defaultMessage
        ? [
            factory.createJsxAttribute(factory.createIdentifier('defaultMessage'), ast
                ? factory.createJsxExpression(undefined, messageASTToTSNode(factory, icu_messageformat_parser_1.parse(msg.defaultMessage)))
                : factory.createStringLiteral(msg.defaultMessage)),
        ]
        : []));
    for (var _i = 0, _a = node.properties; _i < _a.length; _i++) {
        var prop = _a[_i];
        if (ts.isJsxAttribute(prop) &&
            ts.isIdentifier(prop.name) &&
            MESSAGE_DESC_KEYS.includes(prop.name.text)) {
            continue;
        }
        if (ts.isJsxAttribute(prop)) {
            newProps.push(prop);
        }
    }
    return newProps;
}
function extractMessagesFromCallExpression(ts, factory, node, opts, sf) {
    var onMsgExtracted = opts.onMsgExtracted, additionalFunctionNames = opts.additionalFunctionNames;
    if (isMultipleMessageDecl(ts, node)) {
        var _a = node.arguments, arg = _a[0], restArgs = _a.slice(1);
        var descriptorsObj = void 0;
        if (ts.isObjectLiteralExpression(arg)) {
            descriptorsObj = arg;
        }
        else if (ts.isAsExpression(arg) &&
            ts.isObjectLiteralExpression(arg.expression)) {
            descriptorsObj = arg.expression;
        }
        if (descriptorsObj) {
            var properties = descriptorsObj.properties;
            var msgs_1 = properties
                .filter(function (prop) {
                return ts.isPropertyAssignment(prop);
            })
                .map(function (prop) {
                return ts.isObjectLiteralExpression(prop.initializer) &&
                    extractMessageDescriptor(ts, prop.initializer, opts, sf);
            })
                .filter(function (msg) { return !!msg; });
            if (!msgs_1.length) {
                return node;
            }
            console_utils_1.debug('Multiple messages extracted from "%s": %s', sf.fileName, msgs_1);
            if (typeof onMsgExtracted === 'function') {
                onMsgExtracted(sf.fileName, msgs_1);
            }
            var clonedProperties = factory.createNodeArray(properties.map(function (prop, i) {
                if (!ts.isPropertyAssignment(prop) ||
                    !ts.isObjectLiteralExpression(prop.initializer)) {
                    return prop;
                }
                return factory.createPropertyAssignment(prop.name, setAttributesInObject(ts, factory, prop.initializer, {
                    defaultMessage: opts.removeDefaultMessage
                        ? undefined
                        : msgs_1[i].defaultMessage,
                    id: msgs_1[i] ? msgs_1[i].id : '',
                }, opts.ast));
            }));
            var clonedDescriptorsObj = factory.createObjectLiteralExpression(clonedProperties);
            return factory.updateCallExpression(node, node.expression, node.typeArguments, tslib_1.__spreadArray([clonedDescriptorsObj], restArgs));
        }
    }
    else if (isSingularMessageDecl(ts, node, opts.additionalComponentNames || []) ||
        isMemberMethodFormatMessageCall(ts, node, additionalFunctionNames || [])) {
        var _b = node.arguments, descriptorsObj = _b[0], restArgs = _b.slice(1);
        if (ts.isObjectLiteralExpression(descriptorsObj)) {
            var msg = extractMessageDescriptor(ts, descriptorsObj, opts, sf);
            if (!msg) {
                return node;
            }
            console_utils_1.debug('Message extracted from "%s": %s', sf.fileName, msg);
            if (typeof onMsgExtracted === 'function') {
                onMsgExtracted(sf.fileName, [msg]);
            }
            return factory.updateCallExpression(node, node.expression, node.typeArguments, tslib_1.__spreadArray([
                setAttributesInObject(ts, factory, descriptorsObj, {
                    defaultMessage: opts.removeDefaultMessage
                        ? undefined
                        : msg.defaultMessage,
                    id: msg.id,
                }, opts.ast)
            ], restArgs));
        }
    }
    return node;
}
var PRAGMA_REGEX = /^\/\/ @([^\s]*) (.*)$/m;
function getVisitor(ts, ctx, sf, opts) {
    var visitor = function (node) {
        var newNode = ts.isCallExpression(node)
            ? extractMessagesFromCallExpression(ts, ctx.factory, node, opts, sf)
            : ts.isJsxOpeningElement(node) || ts.isJsxSelfClosingElement(node)
                ? extractMessageFromJsxComponent(ts, ctx.factory, node, opts, sf)
                : node;
        return ts.visitEachChild(newNode, visitor, ctx);
    };
    return visitor;
}
function transformWithTs(ts, opts) {
    opts = tslib_1.__assign(tslib_1.__assign({}, DEFAULT_OPTS), opts);
    console_utils_1.debug('Transforming options', opts);
    var transformFn = function (ctx) {
        return function (sf) {
            var pragmaResult = PRAGMA_REGEX.exec(sf.text);
            if (pragmaResult) {
                console_utils_1.debug('Pragma found', pragmaResult);
                var pragma = pragmaResult[1], kvString = pragmaResult[2];
                if (pragma === opts.pragma) {
                    var kvs = kvString.split(' ');
                    var result = {};
                    for (var _i = 0, kvs_1 = kvs; _i < kvs_1.length; _i++) {
                        var kv = kvs_1[_i];
                        var _a = kv.split(':'), k = _a[0], v = _a[1];
                        result[k] = v;
                    }
                    console_utils_1.debug('Pragma extracted', result);
                    if (typeof opts.onMetaExtracted === 'function') {
                        opts.onMetaExtracted(sf.fileName, result);
                    }
                }
            }
            return ts.visitNode(sf, getVisitor(ts, ctx, sf, opts));
        };
    };
    return transformFn;
}
exports.transformWithTs = transformWithTs;
function transform(opts) {
    return transformWithTs(typescript, opts);
}
exports.transform = transform;
